<?php

namespace App\Helpers;

/**
 * Memory Optimization Utility
 * Add this at the top of heavy operation files
 */
class MemoryBooster
{
    public static function init($type = 'default')
    {
        $limits = [
            'default' => '2048M',
            'tournament' => '4096M', 
            'export' => '1024M',
            'ultra' => '8192M'
        ];
        
        $limit = $limits[$type] ?? $limits['default'];
        
        // Set memory limit
        ini_set('memory_limit', $limit);
        
        // Set execution time
        ini_set('max_execution_time', 1800); // 30 minutes
        
        // Optimize PHP settings
        ini_set('max_input_vars', '10000');
        ini_set('post_max_size', '512M');
        ini_set('upload_max_filesize', '256M');
        
        // Enable garbage collection optimization
        gc_enable();
        
        // Force immediate garbage collection
        gc_collect_cycles();
        
        // Disable debug features in production
        if (app()->environment('production')) {
            ini_set('display_errors', 0);
            ini_set('log_errors', 1);
        }
        
        return [
            'memory_limit' => $limit,
            'current_usage' => memory_get_usage(true),
            'peak_usage' => memory_get_peak_usage(true)
        ];
    }
    
    public static function monitor($label = 'Memory Check')
    {
        $usage = memory_get_usage(true);
        $peak = memory_get_peak_usage(true);
        $limit = ini_get('memory_limit');
        
        \Log::info("🔍 {$label}", [
            'current' => self::formatBytes($usage),
            'peak' => self::formatBytes($peak),
            'limit' => $limit,
            'percentage' => round(($usage / self::parseLimit($limit)) * 100, 2) . '%'
        ]);
        
        return compact('usage', 'peak', 'limit');
    }
    
    public static function cleanup()
    {
        // Clear any variables if possible
        if (function_exists('gc_collect_cycles')) {
            $collected = gc_collect_cycles();
            \Log::info("🧹 Garbage collected: {$collected} cycles");
        }
        
        // Clear stat cache
        clearstatcache();
        
        return true;
    }
    
    private static function parseLimit($limit)
    {
        $limit = trim($limit);
        $last = strtolower($limit[strlen($limit) - 1]);
        $value = (int) $limit;
        
        switch ($last) {
            case 'g': return $value * 1024 * 1024 * 1024;
            case 'm': return $value * 1024 * 1024;
            case 'k': return $value * 1024;
            default: return $value;
        }
    }
    
    private static function formatBytes($size)
    {
        $units = ['B', 'KB', 'MB', 'GB'];
        for ($i = 0; $size >= 1024 && $i < 3; $i++) {
            $size /= 1024;
        }
        return round($size, 2) . ' ' . $units[$i];
    }
}

// Auto-initialize with default settings
if (!defined('MEMORY_BOOSTER_INITIALIZED')) {
    MemoryBooster::init('default');
    define('MEMORY_BOOSTER_INITIALIZED', true);
}