<?php

namespace App\Helpers;

use App\Services\ImageOptimizationService;

if (!function_exists('optimized_image')) {
    /**
     * Tạo responsive image với WebP support
     */
    function optimized_image($src, $alt = '', $class = '', $size = 'medium', $lazy = true)
    {
        if (!$src) {
            return '<img src="' . asset('assets/images/default.jpg') . '" alt="' . $alt . '" class="' . $class . '">';
        }

        $imageService = app(ImageOptimizationService::class);
        
        // Convert single path to array format
        if (is_string($src)) {
            $imagePaths = [
                $size => [
                    'webp' => str_replace('.jpg', '.webp', $src),
                    'jpeg' => $src
                ]
            ];
        } else {
            $imagePaths = $src;
        }
        
        $paths = $imageService->getOptimalImagePath($imagePaths, $size);
        
        $lazyClass = $lazy ? 'lazy-image' : '';
        $srcAttr = $lazy ? 'data-src' : 'src';
        $placeholderSrc = $lazy ? 'data:image/svg+xml,%3Csvg xmlns=\'http://www.w3.org/2000/svg\' viewBox=\'0 0 1 1\'%3E%3C/svg%3E' : $paths['jpeg'];
        
        return '<picture class="responsive-image-wrapper ' . $class . '">
            <source type="image/webp" ' . ($lazy ? 'data-srcset' : 'srcset') . '="' . $paths['webp'] . '">
            <source type="image/jpeg" ' . ($lazy ? 'data-srcset' : 'srcset') . '="' . $paths['jpeg'] . '">
            <img 
                src="' . $placeholderSrc . '"
                ' . $srcAttr . '="' . $paths['jpeg'] . '"
                alt="' . $alt . '"
                class="' . $lazyClass . ' ' . $class . '"
                ' . ($lazy ? 'loading="lazy"' : '') . '
                onload="this.classList.add(\'loaded\')"
                onerror="this.src=\'' . asset('assets/images/default.jpg') . '\'; this.classList.add(\'error\');"
            >
        </picture>';
    }
}

if (!function_exists('avatar_image')) {
    /**
     * Tạo avatar image với size responsive
     */
    function avatar_image($src, $alt = '', $size = 'medium', $class = 'player-avatar')
    {
        return optimized_image($src, $alt, $class . ' avatar-' . $size, $size, true);
    }
}

if (!function_exists('venue_image')) {
    /**
     * Tạo venue image với lazy loading
     */
    function venue_image($src, $alt = '', $size = 'large', $class = 'venue-image')
    {
        return optimized_image($src, $alt, $class, $size, true);
    }
}

if (!function_exists('tournament_image')) {
    /**
     * Tạo tournament image
     */
    function tournament_image($src, $alt = '', $size = 'medium', $class = 'tournament-image')
    {
        return optimized_image($src, $alt, $class, $size, true);
    }
}

if (!function_exists('critical_image')) {
    /**
     * Tạo critical image (above-the-fold) - không lazy load
     */
    function critical_image($src, $alt = '', $class = '', $size = 'large')
    {
        return optimized_image($src, $alt, $class . ' critical-image', $size, false);
    }
}

if (!function_exists('preload_image')) {
    /**
     * Tạo preload link cho critical images
     */
    function preload_image($src, $as = 'image')
    {
        if (!$src) return '';
        
        // Convert to WebP if supported
        $webpSrc = str_replace(['.jpg', '.jpeg', '.png'], '.webp', $src);
        
        return '<link rel="preload" as="' . $as . '" href="' . asset($webpSrc) . '" type="image/webp">
                <link rel="preload" as="' . $as . '" href="' . asset($src) . '" type="image/jpeg">';
    }
}