<?php

namespace App\Exports;

use App\Models\Tournament;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\ShouldAutoSize;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Concerns\WithStyles;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;

class ScheduleExport implements FromCollection, ShouldAutoSize, WithHeadings, WithMapping, WithStyles
{
    protected $tournament;

    public function __construct(Tournament $tournament)
    {
        $this->tournament = $tournament;
    }

    /**
     * @return \Illuminate\Support\Collection
     */
    public function collection()
    {
        // Tải tất cả các quan hệ cần thiết để tránh N+1 query
        $rounds = $this->tournament->rounds()->with([
            'matches' => function ($query) {
                $query->with([
                    'pair1.player1',
                    'pair1.player2',
                    'pair1.group.pairs',
                    'pair2.player1',
                    'pair2.player2',
                    'pair2.group',
                ])->orderBy('id');
            },
        ])->orderBy('created_at')->get();

        $rows = collect();

        foreach ($rounds as $round) {
            // Áp dụng logic sắp xếp theo Lượt (circle method) cho các vòng round-robin (vòng bảng)
            if ($round->type === 'round_robin' && $round->matches->count() > 0) {
                $firstMatch = $round->matches->first();
                $group = optional($firstMatch->pair1)->group ?: optional($firstMatch->pair2)->group;

                if ($group) {
                    // Lấy danh sách cặp trong bảng, sắp xếp theo id để cố định thứ tự đầu vào
                    $teamIds = $group->pairs ? $group->pairs->sortBy('id')->pluck('id')->values()->all() : [];

                    // Lập bảng key -> match để truy xuất nhanh
                    $matchMap = [];
                    foreach ($round->matches as $m) {
                        $a = $m->pair1_id;
                        $b = $m->pair2_id;
                        $key = min($a, $b).'-'.max($a, $b);
                        $matchMap[$key] = $m;
                    }

                    $seen = [];
                    $orderedPairsByRound = $this->computeRoundRobinOrder($teamIds);
                    foreach ($orderedPairsByRound as $idx => $pairs) {
                        foreach ($pairs as $pair) {
                            [$a, $b] = $pair;
                            $key = min($a, $b).'-'.max($a, $b);
                            if (isset($matchMap[$key])) {
                                $match = $matchMap[$key];
                                // Gán tên vòng theo đúng hiển thị: Bảng X - Lượt N
                                $match->round_name = 'Bảng '.$group->name.' - Lượt '.($idx + 1);
                                $rows->push($match);
                                $seen[$key] = true;
                            }
                        }
                    }

                    // Thêm các trận chưa xếp được (có thể do thiếu dữ liệu hoặc ngoại lệ)
                    foreach ($round->matches as $m) {
                        $key = min($m->pair1_id, $m->pair2_id).'-'.max($m->pair1_id, $m->pair2_id);
                        if (! isset($seen[$key])) {
                            $m->round_name = 'Bảng '.$group->name;
                            $rows->push($m);
                        }
                    }

                    continue; // đã xử lý xong vòng này
                }
            }

            // Mặc định: giữ nguyên như cũ (playoff hoặc không xác định nhóm)
            foreach ($round->matches as $match) {
                $match->round_name = $round->name;
                $rows->push($match);
            }
        }

        return $rows;
    }

    public function headings(): array
    {
        return [
            'Vòng đấu',
            'Cặp 1',
            'Cặp 2',
            'Tỷ số',
            'Trạng thái',
        ];
    }

    /**
     * @param  mixed  $match
     */
    public function map($match): array
    {
        $pair1Name = $match->pair1 ? ($match->pair1->player1->name.' & '.$match->pair1->player2->name) : 'N/A';
        $pair2Name = $match->pair2 ? ($match->pair2->player1->name.' & '.$match->pair2->player2->name) : 'N/A';

        return [
            $match->round_name,
            $pair1Name,
            $pair2Name,
            $match->score ?: 'Chưa có',
            $match->status,
        ];
    }

    /**
     * @return array
     */
    public function styles(Worksheet $sheet)
    {
        return [
            // Định dạng cho dòng tiêu đề
            1 => [
                'font' => ['bold' => true],
                'fill' => ['fillType' => \PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID, 'startColor' => ['argb' => 'FFD3D3D3']],
            ],
        ];
    }

    /**
     * Circle method (Berger tables): trả về danh sách các lượt, mỗi lượt là mảng các cặp [teamAId, teamBId]
     */
    private function computeRoundRobinOrder(array $teamIds): array
    {
        $teams = array_values($teamIds);
        $rounds = [];

        // Thêm BYE (null) nếu số đội lẻ
        if (count($teams) % 2 === 1) {
            $teams[] = null;
        }

        $n = count($teams);
        if ($n < 2) {
            return $rounds;
        }

        $half = (int) floor($n / 2);
        $rotation = $teams;

        // Số lượt = n - 1 (kể cả khi có BYE)
        for ($r = 0; $r < $n - 1; $r++) {
            $pairs = [];
            for ($i = 0; $i < $half; $i++) {
                $t1 = $rotation[$i];
                $t2 = $rotation[$n - 1 - $i];
                if ($t1 !== null && $t2 !== null) {
                    $pairs[] = [$t1, $t2];
                }
            }
            $rounds[] = $pairs;

            // Xoay: giữ nguyên vị trí 0, đưa phần tử cuối về vị trí 1
            $last = array_pop($rotation);
            array_splice($rotation, 1, 0, [$last]);
        }

        return $rounds;
    }
}
