<?php

namespace App\Exports;

use App\Models\Tournament;
use Illuminate\Support\Collection;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\ShouldAutoSize;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;

class RegistrationsExport implements FromCollection, WithHeadings, WithMapping, ShouldAutoSize
{
    protected Tournament $tournament;

    public function __construct(Tournament $tournament)
    {
        $this->tournament = $tournament;
    }

    public function collection(): Collection
    {
        return $this->tournament->registrations()
            ->with(['player1', 'player2'])
            ->orderBy('created_at')
            ->get();
    }

    public function headings(): array
    {
        return [
            'ID',
            'Ngày đăng ký',
            'Trạng thái',
            'Player 1',
            'Phone 1',
            'Giới tính 1',
            'Điểm đôi 1',
            'Player 2',
            'Phone 2',
            'Giới tính 2',
            'Điểm đôi 2',
            'Tổng điểm trình',
            'Ghi chú',
        ];
    }

    public function map($registration): array
    {
        $player1 = $registration->player1;
        $player2 = $registration->player2;

        $score1 = $player1?->doubles_score;
        $score2 = $player2?->doubles_score;

        $totalScore = collect([$score1, $score2])
            ->filter(fn ($score) => ! is_null($score))
            ->sum();

        return [
            $registration->id,
            optional($registration->created_at)->format('d/m/Y H:i'),
            $this->resolveStatusLabel($registration->status),
            $player1?->name,
            $player1?->phone,
            $player1?->gender,
            $score1,
            $player2?->name,
            $player2?->phone,
            $player2?->gender,
            $score2,
            $totalScore ?: null,
            $registration->reason,
        ];
    }

    protected function resolveStatusLabel(?string $status): string
    {
        return match ($status) {
            'approved' => 'Đã duyệt',
            'pending' => 'Chờ duyệt',
            'rejected' => 'Từ chối',
            default => $status ? ucfirst($status) : 'Không xác định',
        };
    }
}
