<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;

class TestImageOptimization extends Command
{
    protected $signature = 'test:image-optimization';
    protected $description = 'Test image optimization system';

    public function handle()
    {
        $this->info('🧪 Testing Image Optimization System...');
        
        // Test 1: Service loading
        try {
            $service = app(\App\Services\ImageOptimizationService::class);
            $this->info('✅ ImageOptimizationService loaded successfully');
        } catch (\Exception $e) {
            $this->error('❌ Failed to load ImageOptimizationService: ' . $e->getMessage());
            return 1;
        }
        
        // Test 2: Helper functions
        if (function_exists('optimized_image')) {
            $this->info('✅ Helper functions loaded successfully');
        } else {
            $this->error('❌ Helper functions not loaded');
            return 1;
        }
        
        // Test 3: Directory structure
        $directories = [
            'public/assets/avatars/thumb',
            'public/assets/avatars/small', 
            'public/assets/avatars/medium',
            'public/assets/avatars/large',
            'public/assets/avatars/pending'
        ];
        
        foreach ($directories as $dir) {
            if (is_dir(base_path($dir))) {
                $this->info("✅ Directory exists: {$dir}");
            } else {
                $this->warn("⚠️  Directory missing: {$dir}");
            }
        }
        
        // Test 4: Test validation rules
        $this->info('🔍 Testing validation rules...');
        
        try {
            $rules = (new \App\Http\Requests\PlayerRequest)->rules();
            if (str_contains($rules['avatar'], 'webp')) {
                $this->info('✅ PlayerRequest validation updated for WebP');
            } else {
                $this->warn('⚠️  PlayerRequest validation may need WebP support');
            }
        } catch (\Exception $e) {
            $this->error('❌ PlayerRequest validation test failed: ' . $e->getMessage());
        }
        
        // Test 5: Default images
        $defaultImages = [
            'public/assets/images/default.jpg',
            'public/assets/images/default.webp'
        ];
        
        foreach ($defaultImages as $img) {
            if (file_exists(base_path($img))) {
                $this->info("✅ Default image exists: {$img}");
            } else {
                $this->warn("⚠️  Default image missing: {$img}");
            }
        }
        
        $this->info('');
        $this->info('🎉 Image optimization system test completed!');
        $this->info('');
        $this->comment('Next steps:');
        $this->comment('1. Test avatar upload on the frontend');
        $this->comment('2. Check that validation messages are clear');
        $this->comment('3. Verify image optimization is working');
        
        return 0;
    }
}