<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Services\ImageOptimizationService;

class OptimizeExistingImages extends Command
{
    protected $signature = 'images:optimize 
                            {--type=all : Type of images to optimize (avatar, venue, tournament, all)}
                            {--batch=50 : Number of images to process per batch}
                            {--force : Force re-optimization even if optimized versions exist}';

    protected $description = 'Optimize existing images for better performance';

    public function handle(ImageOptimizationService $service)
    {
        $type = $this->option('type');
        $batch = (int) $this->option('batch');
        $force = $this->option('force');
        
        $this->info("🖼️  Starting image optimization...");
        $this->info("Type: {$type}, Batch size: {$batch}");
        
        $types = $type === 'all' ? ['avatar', 'venue', 'tournament'] : [$type];
        $totalProcessed = 0;
        $totalErrors = 0;
        
        foreach ($types as $imageType) {
            $this->info("\n📂 Processing {$imageType} images...");
            
            $results = $service->batchOptimizeExisting($imageType);
            
            $totalProcessed += $results['processed'];
            $totalErrors += $results['errors'];
            
            $this->line("  ✅ Processed: {$results['processed']}");
            $this->line("  ❌ Errors: {$results['errors']}");
        }
        
        $this->info("\n🎉 Optimization completed!");
        $this->info("Total processed: {$totalProcessed}");
        $this->info("Total errors: {$totalErrors}");
        
        // Tạo thư mục cần thiết
        $this->createDirectoryStructure();
        
        // Tạo .htaccess rules cho Apache
        $this->createHtaccessRules();
        
        return 0;
    }
    
    /**
     * Tạo cấu trúc thư mục cho images
     */
    private function createDirectoryStructure(): void
    {
        $types = ['avatars', 'venues', 'tournaments'];
        $sizes = ['thumb', 'small', 'medium', 'large'];
        
        foreach ($types as $type) {
            foreach ($sizes as $size) {
                $path = public_path("assets/{$type}/{$size}");
                if (!is_dir($path)) {
                    mkdir($path, 0755, true);
                    $this->line("📁 Created directory: {$path}");
                }
            }
        }
    }
    
    /**
     * Tạo .htaccess rules cho tối ưu hóa
     */
    private function createHtaccessRules(): void
    {
        $htaccessPath = public_path('assets/.htaccess');
        
        $rules = '# Image optimization rules
# Enable compression
<IfModule mod_deflate.c>
    AddOutputFilterByType DEFLATE image/svg+xml
</IfModule>

# Browser caching
<IfModule mod_expires.c>
    ExpiresActive on
    ExpiresByType image/jpeg "access plus 1 year"
    ExpiresByType image/png "access plus 1 year"
    ExpiresByType image/webp "access plus 1 year"
    ExpiresByType image/gif "access plus 1 year"
    ExpiresByType image/svg+xml "access plus 1 year"
</IfModule>

# Cache control headers
<IfModule mod_headers.c>
    <FilesMatch "\.(jpg|jpeg|png|gif|webp|svg)$">
        Header set Cache-Control "public, max-age=31536000, immutable"
        Header set X-Content-Type-Options nosniff
    </FilesMatch>
</IfModule>

# WebP serving for supporting browsers
<IfModule mod_rewrite.c>
    RewriteEngine On
    
    # Check if browser supports WebP
    RewriteCond %{HTTP_ACCEPT} image/webp
    # Check if WebP version exists
    RewriteCond %{REQUEST_FILENAME}.webp -f
    # Serve WebP version
    RewriteRule (.+)\.(jpe?g|png)$ $1.$2.webp [T=image/webp,E=accept:1,L]
</IfModule>

# Add WebP MIME type
<IfModule mod_mime.c>
    AddType image/webp .webp
</IfModule>';

        file_put_contents($htaccessPath, $rules);
        $this->info("📄 Created .htaccess rules at: {$htaccessPath}");
    }
}