<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;

class OptimizeDatabaseCommand extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'db:optimize-tournament';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Optimize database for tournament operations by adding indexes and analyzing performance';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('Starting database optimization for tournament operations...');

        // Add indexes for frequently queried columns
        $this->addIndexes();

        // Clean up old logs and temporary data
        $this->cleanupOldData();

        // Optimize tables
        $this->optimizeTables();

        $this->info('Database optimization completed successfully!');
    }

    /**
     * Add indexes for better performance
     */
    private function addIndexes()
    {
        $this->info('Adding database indexes...');

        $indexes = [
            'tournaments' => [
                ['status'],
                ['created_at'],
                ['venue_id']
            ],
            'registrations' => [
                ['tournament_id', 'status'],
                ['player1_id'],
                ['player2_id'],
                ['created_at']
            ],
            'tournament_groups' => [
                ['tournament_id'],
                ['name']
            ],
            'group_pairs' => [
                ['group_id'],
                ['player1_id'],
                ['player2_id']
            ],
            'tournament_rounds' => [
                ['tournament_id', 'type'],
                ['name'],
                ['created_at']
            ],
            'matches' => [
                ['tournament_id', 'status'],
                ['tournament_round_id'],
                ['pair1_id'],
                ['pair2_id'],
                ['created_at'],
                ['updated_at']
            ],
            'players' => [
                ['name'],
                ['phone'],
                ['created_at']
            ]
        ];

        foreach ($indexes as $table => $tableIndexes) {
            if (!Schema::hasTable($table)) {
                continue;
            }

            foreach ($tableIndexes as $columns) {
                $indexName = $table . '_' . implode('_', $columns) . '_index';
                
                try {
                    // Check if index already exists (basic check)
                    $existingIndexes = DB::select("SHOW INDEX FROM {$table}");
                    $indexExists = collect($existingIndexes)->pluck('Key_name')->contains($indexName);
                    
                    if (!$indexExists) {
                        Schema::table($table, function ($tableSchema) use ($columns, $indexName) {
                            $tableSchema->index($columns, $indexName);
                        });
                        $this->line("✓ Added index on {$table}(" . implode(', ', $columns) . ")");
                    } else {
                        $this->line("- Index already exists on {$table}(" . implode(', ', $columns) . ")");
                    }
                } catch (\Exception $e) {
                    $this->warn("Failed to add index on {$table}: " . $e->getMessage());
                }
            }
        }
    }

    /**
     * Clean up old data
     */
    private function cleanupOldData()
    {
        $this->info('Cleaning up old data...');

        // Clean up activity logs older than 3 months
        $cleaned = DB::table('activity_log')
            ->where('created_at', '<', now()->subMonths(3))
            ->delete();
        
        if ($cleaned > 0) {
            $this->line("✓ Cleaned up {$cleaned} old activity log entries");
        }

        // Clean up failed jobs older than 1 week
        $cleanedJobs = DB::table('failed_jobs')
            ->where('failed_at', '<', now()->subWeek())
            ->delete();
            
        if ($cleanedJobs > 0) {
            $this->line("✓ Cleaned up {$cleanedJobs} old failed jobs");
        }
    }

    /**
     * Optimize tables
     */
    private function optimizeTables()
    {
        $this->info('Optimizing tables...');

        $tables = [
            'tournaments', 'registrations', 'tournament_groups', 
            'group_pairs', 'tournament_rounds', 'matches', 'players'
        ];

        foreach ($tables as $table) {
            try {
                DB::statement("OPTIMIZE TABLE {$table}");
                $this->line("✓ Optimized table {$table}");
            } catch (\Exception $e) {
                $this->warn("Failed to optimize table {$table}: " . $e->getMessage());
            }
        }
    }
}