<?php

namespace App\Console\Commands;

use App\Models\Player;
use App\Models\User;
use Illuminate\Console\Command;

class MigrateAvatarsToAssets extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'avatars:migrate-to-assets';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Migrate avatars from storage to public/assets';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('Starting avatar migration from storage to assets...');

        // Ensure directories exist
        if (!file_exists(public_path('assets/avatars'))) {
            mkdir(public_path('assets/avatars'), 0755, true);
        }
        if (!file_exists(public_path('assets/avatars/pending'))) {
            mkdir(public_path('assets/avatars/pending'), 0755, true);
        }

        // Migrate users
        $users = User::whereNotNull('avatar')->where('avatar', 'like', 'avatars/%')->get();
        $this->info("Found {$users->count()} users with storage avatars to migrate.");

        $migratedUsers = 0;
        foreach ($users as $user) {
            $oldPath = $user->avatar; // e.g., "avatars/filename.jpg"
            $storagePath = storage_path('app/public/' . $oldPath);
            
            if (file_exists($storagePath)) {
                $filename = basename($oldPath);
                $newFilename = 'user_' . $user->id . '_' . time() . '_' . $filename;
                $assetsPath = public_path('assets/avatars/' . $newFilename);
                
                if (copy($storagePath, $assetsPath)) {
                    $user->avatar = 'assets/avatars/' . $newFilename;
                    $user->save();
                    $this->line("Migrated user {$user->id}: {$oldPath} -> assets/avatars/{$newFilename}");
                    $migratedUsers++;
                    
                    // Delete old file
                    unlink($storagePath);
                } else {
                    $this->error("Failed to migrate user {$user->id}: {$oldPath}");
                }
            } else {
                $this->warn("File not found for user {$user->id}: {$storagePath}");
                // Update path to null since file doesn't exist
                $user->avatar = null;
                $user->save();
            }
        }

        // Migrate players (both avatar and avatar_pending)
        $players = Player::where(function($query) {
            $query->where('avatar', 'like', 'avatars/%')
                  ->orWhere('avatar_pending', 'like', 'avatars/%');
        })->get();
        $this->info("Found {$players->count()} players with storage avatars to migrate.");

        $migratedPlayers = 0;
        foreach ($players as $player) {
            $updated = false;
            
            // Migrate main avatar
            if ($player->avatar && str_starts_with($player->avatar, 'avatars/')) {
                $oldPath = $player->avatar;
                $storagePath = storage_path('app/public/' . $oldPath);
                
                if (file_exists($storagePath)) {
                    $filename = basename($oldPath);
                    $newFilename = 'player_' . $player->id . '_' . time() . '_' . $filename;
                    $assetsPath = public_path('assets/avatars/' . $newFilename);
                    
                    if (copy($storagePath, $assetsPath)) {
                        $player->avatar = 'assets/avatars/' . $newFilename;
                        $this->line("Migrated player {$player->id} avatar: {$oldPath} -> assets/avatars/{$newFilename}");
                        unlink($storagePath);
                        $updated = true;
                    } else {
                        $this->error("Failed to migrate player {$player->id} avatar: {$oldPath}");
                    }
                } else {
                    $this->warn("Avatar file not found for player {$player->id}: {$storagePath}");
                    $player->avatar = null;
                    $updated = true;
                }
            }
            
            // Migrate pending avatar
            if ($player->avatar_pending && str_starts_with($player->avatar_pending, 'avatars/')) {
                $oldPath = $player->avatar_pending;
                $storagePath = storage_path('app/public/' . $oldPath);
                
                if (file_exists($storagePath)) {
                    $filename = basename($oldPath);
                    $newFilename = 'player_' . $player->id . '_' . time() . '_pending_' . $filename;
                    $assetsPath = public_path('assets/avatars/pending/' . $newFilename);
                    
                    if (copy($storagePath, $assetsPath)) {
                        $player->avatar_pending = 'assets/avatars/pending/' . $newFilename;
                        $this->line("Migrated player {$player->id} pending avatar: {$oldPath} -> assets/avatars/pending/{$newFilename}");
                        unlink($storagePath);
                        $updated = true;
                    } else {
                        $this->error("Failed to migrate player {$player->id} pending avatar: {$oldPath}");
                    }
                } else {
                    $this->warn("Pending avatar file not found for player {$player->id}: {$storagePath}");
                    $player->avatar_pending = null;
                    $player->avatar_pending_requested_at = null;
                    $updated = true;
                }
            }
            
            if ($updated) {
                $player->save();
                $migratedPlayers++;
            }
        }

        $this->info('');
        $this->info('Migration completed!');
        $this->info("Users migrated: {$migratedUsers}");
        $this->info("Players migrated: {$migratedPlayers}");
        $this->info('All avatars are now using the assets system.');
        
        return self::SUCCESS;
    }
}
