<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class EmergencyCleanupCommand extends Command
{
    /**
     * The name and signature of the console command.
     */
    protected $signature = 'emergency:cleanup {--force : Force cleanup without confirmation}';

    /**
     * The console command description.
     */
    protected $description = 'Emergency cleanup to free memory and resources';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('🚨 Emergency Cleanup Starting...');

        if (!$this->option('force')) {
            if (!$this->confirm('This will clear all caches and optimize the database. Continue?')) {
                return;
            }
        }

        // 1. Clear all caches
        $this->clearAllCaches();

        // 2. Clear logs
        $this->clearOldLogs();

        // 3. Optimize database
        $this->optimizeDatabase();

        // 4. Clear sessions
        $this->clearOldSessions();

        // 5. Force garbage collection
        $this->forceGarbageCollection();

        // 6. Clear opcache if available
        $this->clearOpcache();

        $this->info('✅ Emergency cleanup completed!');
        $this->displayMemoryStats();
    }

    /**
     * Clear all Laravel caches
     */
    private function clearAllCaches(): void
    {
        $this->info('Clearing all caches...');

        try {
            // Clear application cache
            Cache::flush();
            $this->line('✓ Application cache cleared');

            // Clear config cache
            $this->call('config:clear');
            $this->line('✓ Config cache cleared');

            // Clear route cache
            $this->call('route:clear');
            $this->line('✓ Route cache cleared');

            // Clear view cache
            $this->call('view:clear');
            $this->line('✓ View cache cleared');

            // Clear compiled services
            $this->call('clear-compiled');
            $this->line('✓ Compiled services cleared');

        } catch (\Exception $e) {
            $this->warn('Cache clearing error: ' . $e->getMessage());
        }
    }

    /**
     * Clear old logs to free disk space
     */
    private function clearOldLogs(): void
    {
        $this->info('Clearing old logs...');

        try {
            $logPath = storage_path('logs');
            $files = glob($logPath . '/*.log');
            
            $clearedSize = 0;
            $clearedCount = 0;

            foreach ($files as $file) {
                if (is_file($file)) {
                    $size = filesize($file);
                    
                    // Keep today's log, truncate others
                    if (basename($file) === 'laravel.log') {
                        $content = file_get_contents($file);
                        $lines = explode("\n", $content);
                        
                        // Keep only last 1000 lines
                        if (count($lines) > 1000) {
                            $truncated = implode("\n", array_slice($lines, -1000));
                            file_put_contents($file, $truncated);
                            $clearedSize += $size - filesize($file);
                            $clearedCount++;
                        }
                    } else {
                        // Delete old log files
                        $modTime = filemtime($file);
                        if ($modTime < strtotime('-7 days')) {
                            $clearedSize += $size;
                            unlink($file);
                            $clearedCount++;
                        }
                    }
                }
            }

            $this->line("✓ Cleared {$clearedCount} log files (" . number_format($clearedSize / 1024 / 1024, 2) . " MB)");

        } catch (\Exception $e) {
            $this->warn('Log clearing error: ' . $e->getMessage());
        }
    }

    /**
     * Optimize database tables
     */
    private function optimizeDatabase(): void
    {
        $this->info('Optimizing database...');

        try {
            $tables = [
                'tournaments', 'registrations', 'tournament_groups', 
                'group_pairs', 'tournament_rounds', 'matches', 
                'players', 'cache', 'sessions', 'activity_log'
            ];

            foreach ($tables as $table) {
                DB::statement("OPTIMIZE TABLE {$table}");
            }

            $this->line('✓ Database tables optimized');

            // Clean up old activity logs
            $deleted = DB::table('activity_log')
                ->where('created_at', '<', now()->subDays(30))
                ->delete();

            if ($deleted > 0) {
                $this->line("✓ Deleted {$deleted} old activity log entries");
            }

        } catch (\Exception $e) {
            $this->warn('Database optimization error: ' . $e->getMessage());
        }
    }

    /**
     * Clear old sessions
     */
    private function clearOldSessions(): void
    {
        $this->info('Clearing old sessions...');

        try {
            $deleted = DB::table('sessions')
                ->where('last_activity', '<', time() - 86400) // Older than 24 hours
                ->delete();

            $this->line("✓ Deleted {$deleted} old sessions");

        } catch (\Exception $e) {
            $this->warn('Session clearing error: ' . $e->getMessage());
        }
    }

    /**
     * Force garbage collection
     */
    private function forceGarbageCollection(): void
    {
        $this->info('Running garbage collection...');

        $before = memory_get_usage(true);

        // Run multiple GC cycles
        for ($i = 0; $i < 5; $i++) {
            gc_collect_cycles();
        }

        $after = memory_get_usage(true);
        $freed = $before - $after;

        if ($freed > 0) {
            $this->line('✓ Freed ' . number_format($freed / 1024 / 1024, 2) . ' MB of memory');
        } else {
            $this->line('✓ Garbage collection completed');
        }
    }

    /**
     * Clear OPcache if available
     */
    private function clearOpcache(): void
    {
        if (function_exists('opcache_reset')) {
            opcache_reset();
            $this->line('✓ OPcache cleared');
        }

        if (function_exists('opcache_compile_file')) {
            $this->line('✓ OPcache available');
        }
    }

    /**
     * Display current memory statistics
     */
    private function displayMemoryStats(): void
    {
        $this->info('📊 Current Memory Stats:');
        $this->line('Memory Usage: ' . number_format(memory_get_usage(true) / 1024 / 1024, 2) . ' MB');
        $this->line('Peak Memory: ' . number_format(memory_get_peak_usage(true) / 1024 / 1024, 2) . ' MB');
        $this->line('Memory Limit: ' . ini_get('memory_limit'));
    }
}